/**
* \file: errmemd_backend_channel.c
*
* Implementation of backend to dump to a given channle.
* 
* This file implements the interface functions for data streams.
*
* \component: errmemd
*
* \author: Markus Kretschmann (mkretschmann@de.adit-jv.com)
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
* <history item>
*/
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <syslog.h>
#include <unistd.h>

#include "errmem_backend.h"
#include "errmem_lib.h"

/*
 * Implementation of an error memory backend that dumps error memory
 * messages to stdout or stderr.
 */

typedef struct ErrmemBackendChannel {
	ErrmemBackend_t backend; // error memory backend interface
	int32_t         type;    // type of backend "storage"
	FILE*           hd;      // output channel
	int32_t         debug;   // writes addtional debug output
} ErrmemBackendChannel_t;

static unsigned get_type(ErrmemBackend_t* backend)
{
	if(backend)
		return ((ErrmemBackendChannel_t*)backend)->type;
	return 0;
}

// info - show information about this backend
static void info(struct ErrmemBackend* backend, int fd)
{
	if (get_type(backend) & ERRMEM_STD_OUT)
		dprintf(fd, "error memory daemon has created interface to standard out\n");
	else if (get_type(backend) & ERRMEM_STD_ERR)
		dprintf(fd, "error memory daemon has created interface to standard error\n");
	else
		dprintf(fd, "no info available\n");
}

// store - store an error memory message
static unsigned store(
	struct ErrmemBackend* backend, int len,
	struct errmem_message* msg)
{
	int32_t i = 0;
	len = len;
	if (backend) {
		/* PRQA: Lint Message 648: We explicitly want to create a mask*/
		/*lint -save -e648 */
		unsigned owner = ERRMEM_GET_VM(msg->internal.flags);
		unsigned flags = ERRMEM_MASK_VM(msg->internal.flags);
		/*lint -restore */
		if (((ErrmemBackendChannel_t*)backend)->debug)
			fprintf(((ErrmemBackendChannel_t*)backend)->hd, "errmem_client: type = %d seqnum = %d clock = %u %u"
					" offset = %d flags = %d owner = %d crc = %d storage = %d length = %d\n",
					msg->type, msg->internal.seqnum, msg->internal.local_clock[0],
					msg->internal.local_clock[1], msg->internal.offset,
					flags, owner, msg->internal.crc, msg->internal.storage,
					msg->length);
		if (msg->type == ERRMEM_TYPE_ASCII) {
			switch(owner) {
			case OWNER_NONE:
				fprintf(((ErrmemBackendChannel_t*)backend)->hd,
						"%s",
						(char*)&msg->message);
				break;
			case OWNER_VMM:
				fprintf(((ErrmemBackendChannel_t*)backend)->hd,
						VMM_STRING"%s",
						(char*)&msg->message);
				break;
			default:
				fprintf(((ErrmemBackendChannel_t*)backend)->hd,
						VM_STRING"%s",
						owner - 2,
						(char*)&msg->message);
			}
		} else {
			for (i = 0; i < msg->length; i++)
				fprintf(((ErrmemBackendChannel_t*)backend)->hd, "%02x ", msg->message[i]);
			fprintf(((ErrmemBackendChannel_t*)backend)->hd, "\n");
		}	
	}
	return msg->internal.seqnum;
}

// destroy - free resources
static void destroy(struct ErrmemBackend* backend)
{
	ErrmemBackendChannel_t* dev = (ErrmemBackendChannel_t*)backend;
	
	if (dev)
		free(dev);
}

// create_channel_structure - creates channel strture for DLT/STD_OUT/STD_ERR
ErrmemBackend_t* create_channel_structure(ErrmemBackendDev_t* p_dev)
{
	ErrmemBackendChannel_t* dev = NULL;

	if (p_dev) {
		dev = (ErrmemBackendChannel_t*)get_mem(sizeof(ErrmemBackendChannel_t));

		if (!dev)
			syslog(LOG_CRIT, "%s %s",
				   "Channel-Backend could not be created - error: ",
				   strerror(ENOMEM));	
		else {
			dev->backend.info     = info;
			dev->backend.store    = store;
			dev->backend.destroy  = destroy;
			dev->backend.get_type = get_type;
			dev->type             = p_dev->type;
			if (dev->type & ERRMEM_STD_OUT)
				dev->hd = stdout;
			if (dev->type & ERRMEM_STD_ERR)
				dev->hd = stderr;
			dev->debug            = p_dev->debug;
		}
	} else {
		syslog(LOG_CRIT, "%s %s",
			   "Channel-Backend: Parameter check failed - create_channel_structure",
			   strerror(EINVAL));
	}	

	if (dev)
		return &dev->backend;
	else
		return NULL;
}
